import dochint
import unittest

import html

class TestCases(unittest.TestCase):
    def test_single_chars(self):
        text = '\\\\ \\< \\> \\& \\" \\\''
        processed = '\\ &lt; &gt; &amp; &quot; &apos;'
        self.assertEqual(dochint.process_text(text), processed)

    def test_escaped_newline(self):
        text = '12\\\n34'
        processed = '1234'
        self.assertEqual(dochint.process_text(text), processed)

    def test_invalid_single_char(self):
        with self.assertRaises(dochint.MacroCommandError):
            dochint.process_text('\\%ab')

    def test_macro_empty(self):
        with self.assertRaises(dochint.MacroEmptyError):
            dochint.process_text('12\\<34\\')
    
    def test_macro_invalid_name(self):
        with self.assertRaises(dochint.MacroCommandError):
            dochint.process_text('\\thisisnotarealmacro')

    def test_verbatim1(self):
        text = '\\verb|<p>There are <em>four</em> lights!</p>|'
        processed = html.escape('<p>There are <em>four</em> lights!</p>')
        self.assertEqual(dochint.process_text(text), processed)

    def test_verbatim2(self):
        text = '12 \\verbatim~<~ 34'
        processed = '12 &lt; 34'
        self.assertEqual(dochint.process_text(text), processed)

    def test_verbatim_incomplete1(self):
        with self.assertRaises(dochint.IncompleteMacroError):
            dochint.process_text('12\\verbatim')

    def test_verbatim_incomplete2(self):
        with self.assertRaises(dochint.IncompleteMacroError):
            dochint.process_text('12 \\verb|<')

    def test_extra_macros(self):
        text = '<em>\\title</em> by \\author'
        processed = '<em>Moby-Dick</em> by Herman Melville'
        extra_macros = {'title': 'Moby-Dick', 'author': 'Herman Melville'}
        self.assertEqual(dochint.process_text(text, extra_macros=extra_macros),
                         processed)

    def test_alternative_macro_prefix1(self):
        text = '@@@< maps to @@&amp;lt;'
        processed = '@&lt; maps to @&amp;lt;'
        self.assertEqual(dochint.process_text(text, prefix='@'), processed)

    def test_alternative_macro_prefix2(self):
        text = '######< maps to ####&amp;lt;'
        processed = '##&lt; maps to ##&amp;lt;'
        self.assertEqual(dochint.process_text(text, prefix='##'), processed)

    def test_math1(self):
        text = '\\math{E=mc^2} is'
        processed = '<math alttext="E=mc^2" xmlns="http://www.w3.org/1998/Math/MathML" display="inline"><mrow><mi>E</mi><mo>&#x0003D;</mo><mi>m</mi><msup><mi>c</mi><mn>2</mn></msup></mrow></math> is'
        self.assertEqual(dochint.process_text(text), processed)

    def test_math2(self):
        text = '\\dmath{E=mc^2} is'
        processed = '<div class=\'mathsblock\'><math alttext="E=mc^2" xmlns="http://www.w3.org/1998/Math/MathML" display="block"><mrow><mi>E</mi><mo>&#x0003D;</mo><mi>m</mi><msup><mi>c</mi><mn>2</mn></msup></mrow></math></div> is'
        self.assertEqual(dochint.process_text(text), processed)

    def test_math3(self):
        text = '\\math{\\frac{1}{2}}'
        processed = '<math alttext="\\frac{1}{2}" xmlns="http://www.w3.org/1998/Math/MathML" display="inline"><mrow><mfrac><mrow><mn>1</mn></mrow><mrow><mn>2</mn></mrow></mfrac></mrow></math>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_math4(self):
        text = '\\math{\\{}'
        processed = '<math alttext="\\{" xmlns="http://www.w3.org/1998/Math/MathML" display="inline"><mrow><mo stretchy="false">&#x0007B;</mo></mrow></math>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_math_incomplete_arg(self):
        with self.assertRaises(dochint.IncompleteMacroError):
            dochint.process_text('\\math{a^2+b^2=c^2')

    def test_math_missing_arg(self):
        with self.assertRaises(dochint.IncompleteMacroError):
            dochint.process_text('\\math is')

    def test_math_latex_error(self):
        with self.assertRaises(dochint.LaTeXMathsError):
            dochint.process_text('\\math{\\frac}')

    def test_custom_opts_args1(self):
        def func(*args: str, opts: list[str]=[], **kwargs):
            return f'Options: {opts}, arguments: {args}'
        text = '\\custom[1]{2}{3}{4}'
        processed = 'Options: [\'1\'], arguments: (\'2\', \'3\', \'4\')'
        custom_macro = dochint.ArgsMacro(func, '[i][i][i]{i}{i}{i}')
        extra_macros = {'custom': custom_macro}
        self.assertEqual(dochint.process_text(text, extra_macros=extra_macros),
                         processed)

    def test_custom_opts_args2(self):
        def func(*args: str, opts: list[str]=[], **kwargs):
            return f'Options: {opts}, arguments: {args}'
        text = '\\custom  {2}    {3}   {4}'
        processed = 'Options: [], arguments: (\'2\', \'3\', \'4\')'
        custom_macro = dochint.ArgsMacro(func, '[i][i][i]{i}{i}{i}')
        extra_macros = {'custom': custom_macro}
        self.assertEqual(dochint.process_text(text, extra_macros=extra_macros),
                         processed)

    def test_custom_opts_args_too_many_opts(self):
        def func(*args: str, opts: list[str]=[], **kwargs):
            return f'Options: {opts}, arguments: {args}'
        text = '\\custom[1][2]{3}{4}{5}'
        extra_macros = {'custom': dochint.ArgsMacro(func, '[i]{i}{i}{i}')}
        with self.assertRaises(dochint.IncompleteMacroError):
            dochint.process_text(text, extra_macros=extra_macros)

    def test_custom_opts_args_too_many_args(self):
        def func(*args: str, opts: list[str]=[], **kwargs):
            return f'Options: {opts}, arguments: {args}'
        text = '\\custom[1][2]{3}{4}{5}'
        processed = 'Options: [\'1\', \'2\'], arguments: (\'3\', \'4\'){5}'
        extra_macros = {'custom': dochint.ArgsMacro(func, '[i][i]{i}{i}')}
        self.assertEqual(dochint.process_text(text, extra_macros=extra_macros),
                         processed)

    def test_crossref_1(self):
        text = '<span id=\\id{ref1}>test</span>\n\\ref{ref1}'
        processed = '<span id="ref1">test</span>\n<a href="#ref1">1</a>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_crossref_2(self):
        text = '<span id=\\id{ref1}>test</span>'
        text += '\n<span id=\\id{ref2}>test</span>'
        text += '\n\\ref{ref1},\\ref{ref2}'
        processed = '<span id="ref1">test</span>'
        processed += '\n<span id="ref2">test</span>'
        processed += '\n<a href="#ref1">1</a>,<a href="#ref2">2</a>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_crossref_labelled(self):
        text = '<span id=\\id[test]{ref1}>test</span>\n\\ref{ref1}'
        processed = '<span id="ref1">test</span>\n<a href="#ref1">test</a>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_crossref_namespaces(self):
        text = '<span id=\\id{abc:1}>test</span>'
        text += '\n<span id=\\id{def:1}>test</span>'
        text += '\n\\ref{abc:1},\\ref{def:1}'
        processed = '<span id="abc:1">test</span>'
        processed += '\n<span id="def:1">test</span>'
        processed += '\n<a href="#abc:1">1</a>,<a href="#def:1">1</a>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_crossref_before(self):
        text = '\\ref{ref1}\n<span id=\\id{ref1}>test</span>'
        processed = '<a href="#ref1">1</a>\n<span id="ref1">test</span>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_crossref_missing(self):
        text = '<span id=\\id{ref1}>test</span>\n\\ref{ref2}'
        with self.assertRaises(dochint.CrossrefNotFoundError):
            dochint.process_text(text)

    def test_crossref_duplicated(self):
        text = '<span id=\\id{ref1}>test</span>'
        text += '\n<span id=\\id{ref1}>test</span>'
        with self.assertRaises(dochint.CrossrefExistsError):
            dochint.process_text(text)

    def test_equation_numbered(self):
        text = '\\equation{eqn:1}{E=mc^2}'
        processed = '<figure class=\'equation\' id="eqn:1"><math alttext="E=mc^2" xmlns="http://www.w3.org/1998/Math/MathML" display="block"><mrow><mi>E</mi><mo>&#x0003D;</mo><mi>m</mi><msup><mi>c</mi><mn>2</mn></msup></mrow></math><figcaption>(1)</figcaption></figure>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_equation_labelled(self):
        text = '\\equation[A]{eqn:1}{E=mc^2}'
        processed = '<figure class=\'equation\' id="eqn:1"><math alttext="E=mc^2" xmlns="http://www.w3.org/1998/Math/MathML" display="block"><mrow><mi>E</mi><mo>&#x0003D;</mo><mi>m</mi><msup><mi>c</mi><mn>2</mn></msup></mrow></math><figcaption>(A)</figcaption></figure>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_equation_and_ref(self):
        text = 'See Equation \\ref{eqn:1}\n\\equation{eqn:1}{E=mc^2}'
        processed = 'See Equation <a href="#eqn:1">1</a>\n<figure class=\'equation\' id="eqn:1"><math alttext="E=mc^2" xmlns="http://www.w3.org/1998/Math/MathML" display="block"><mrow><mi>E</mi><mo>&#x0003D;</mo><mi>m</mi><msup><mi>c</mi><mn>2</mn></msup></mrow></math><figcaption>(1)</figcaption></figure>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_citations(self):
        with open('test/citations-text.html') as f:
            text = f.read()
        with open('test/citations-processed.html') as f:
            processed = f.read()
        self.maxDiff = None
        self.assertEqual(dochint.process_text(text), processed)

    def test_citation_missing(self):
        text = '\\cite{missing_citation}\\bibliography'
        with self.assertRaises(dochint.CitationError):
            dochint.process_text(text)

    def test_no_bibliography(self):
        text = '\\cite{a_citation}'
        with self.assertRaises(dochint.CitationError):
            dochint.process_text(text)

    def test_bad_bibtex(self):
        with self.assertRaises(dochint.BibTeXError):
            dochint.process_text('\\addbibtextext{@article{}}')

    def test_footnotes(self):
        text = 'Some text\\footnote{A footnote.}. '
        text += 'Some more text\\footnote{Another footnote.}.\n'
        text += '\\footnotes'
        processed = 'Some text<sup><a href="#_footnote_1_1">1</a></sup>. '
        processed += 'Some more text<sup><a href="#_footnote_1_2">2</a></sup>.\n'
        processed += '<p id="_footnote_1_1"><sup>1</sup>A footnote.</p>\n'
        processed += '<p id="_footnote_1_2"><sup>2</sup>Another footnote.</p>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_footnote_error(self):
        with self.assertRaises(dochint.FootnoteError):
            dochint.process_text('\\footnote{A footnote.}')

    def test_footnote_recursive(self):
        text = 'Some text\\footnote{A footnote.}. '
        text += 'Some more text\\footnote{12 \\< 34}.\n'
        text += '\\footnotes'
        processed = 'Some text<sup><a href="#_footnote_1_1">1</a></sup>. '
        processed += 'Some more text<sup><a href="#_footnote_1_2">2</a></sup>.\n'
        processed += '<p id="_footnote_1_1"><sup>1</sup>A footnote.</p>\n'
        processed += '<p id="_footnote_1_2"><sup>2</sup>12 &lt; 34</p>'
        self.assertEqual(dochint.process_text(text), processed)

    def test_multipage(self):
        with open('test/multipage_p1-text.html') as f:
            text_p1 = f.read()
        with open('test/multipage_p2-text.html') as f:
            text_p2 = f.read()
        with open('test/multipage_p1-processed.html') as f:
            processed_p1 = f.read()
        with open('test/multipage_p2-processed.html') as f:
            processed_p2 = f.read()
        self.maxDiff = None
        self.assertEqual(
            dochint.process_texts([('p1.html', text_p1),
                                   ('p2.html', text_p2)]),
            {'p1.html': processed_p1, 'p2.html': processed_p2})

    def test_set_numbering(self):
        with open('test/set_numbering_p1-text.html') as f:
            text_p1 = f.read()
        with open('test/set_numbering_p2-text.html') as f:
            text_p2 = f.read()
        with open('test/set_numbering_p3-text.html') as f:
            text_p3 = f.read()
        with open('test/set_numbering_p4-text.html') as f:
            text_p4 = f.read()
        with open('test/set_numbering_p1-processed.html') as f:
            processed_p1 = f.read()
        with open('test/set_numbering_p2-processed.html') as f:
            processed_p2 = f.read()
        with open('test/set_numbering_p3-processed.html') as f:
            processed_p3 = f.read()
        with open('test/set_numbering_p4-processed.html') as f:
            processed_p4 = f.read()
        texts = [('p1.html', text_p1),
                 ('p2.html', text_p2),
                 ('p3.html', text_p3),
                 ('p4.html', text_p4)]
        processed = {'p1.html': processed_p1, 'p2.html': processed_p2,
                     'p3.html': processed_p3, 'p4.html': processed_p4}
        numberings = {'p2.html': 1, 'p4.html': 'a'}
        self.maxDiff = None
        self.assertEqual(
            dochint.process_texts(texts, numberings=numberings), processed)

if __name__ == '__main__':
    unittest.main()
