from . import file_helper, templates

import datetime
import os
import sys
from xml.sax.saxutils import escape as xml_escape
from xml.sax.saxutils import quoteattr
import zipfile

prog = os.path.basename(sys.argv[0])

def timestamp_now():
    t = datetime.datetime.now(datetime.timezone.utc)
    return t.strftime('%Y-%m-%dT%H:%M:%SZ')

def manifest_items(opts):
    items = []

    # toc item
    item = f'<item href={quoteattr(opts.toc)} id=\'toc\' '
    item += f'media-type=\'application/xhtml+xml\' properties=\'nav\'/>'
    items.append(item)

    def manifest_item(id_, name):
        mime = file_helper.get_mimetype(name)
        name_a = quoteattr(name)
        mime_a = quoteattr(mime)
        item = f'<item href={name_a} id=\'{id_}\' media-type={mime_a}/>'
        return item

    for i, name in enumerate(opts.spine):
        items.append(manifest_item(f'spine{i}', name))

    for i, name in enumerate(opts.resources):
        items.append(manifest_item(f'res{i}', name))

    for i, name in enumerate(opts.stylesheets):
        items.append(manifest_item(f'style{i}', name))

    return '\n'.join(items)

def spine_items(opts):
    items = []
    for i, name in enumerate(opts.spine):
        id_ = f'spine{i}'
        item = f'<itemref idref=\'{id_}\'></itemref>'
        items.append(item)

    return '\n'.join(items)

def write_package_opf(z, opts):
    timestamp = timestamp_now()

    dc_values = [('dc:coverage', opts.dc_coverage),
                 ('dc:creator', opts.dc_creator),
                 ('dc:date', timestamp if opts.dc_date_now
                             else opts.dc_date),
                 ('dc:description', opts.dc_description),
                 ('dc:publisher', opts.dc_publisher),
                 ('dc:rights', opts.dc_rights),
                 ('dc:source', opts.dc_source),
                 ('dc:subject', opts.dc_subject),
                 ('dc:type', opts.dc_type)]
    dc_values += [('dc:contributor', c) for c in opts.dc_contributor]
    dc_values += [('dc:relation', c) for c in opts.dc_relation]

    dc_extra = '\n'.join((f'<{tag}>{xml_escape(value)}</{tag}>'
                          for (tag, value) in dc_values
                          if value is not None))

    package = templates.package_opf.format(
        dc_identifier=xml_escape(opts.dc_identifier),
        dc_title=xml_escape(opts.dc_title),
        dc_language=xml_escape(opts.dc_language),
        dc_format=xml_escape(templates.mimetype),
        dc_extra=dc_extra,
        dcterms_modified=xml_escape(timestamp),
        manifestitems=manifest_items(opts),
        spineitems=spine_items(opts),
    )

    f = z.open('package.opf', 'w')
    f.write(package.encode('utf-8'))
    f.close()

def write_resource_files(z, opts):
    for name in opts.resources:
        fpath = f'{opts.source_dir}/{name}'
        z.write(fpath, name)

def write_templated_xhtml(z, name, opts):
    fpath = f'{opts.source_dir}/{name}'
    with open(fpath, 'r') as f:
        file = f.read()

    stylelink_fmt = '<link rel=\'stylesheet\' href={name_a}/>'
    stylelinks = '\n'.join(stylelink_fmt.format(name_a=quoteattr(name))
                           for name in opts.stylesheets)

    html = opts.template_html.format(dc_title=xml_escape(opts.dc_title),
                                     stylelinks=stylelinks, file=file)
    xhtml = opts.template_xhtml.format(html=html,
        dc_title=xml_escape(opts.dc_title), stylelinks=stylelinks, file=file)

    f = z.open(name, 'w')
    f.write(xhtml.encode('utf-8'))
    f.close()

def write_spine_files(z, opts):
    for name in opts.spine:
        fpath = f'{opts.source_dir}/{name}'
        if file_helper.get_extension(name).lower() == 'svg':
            z.write(fpath, name)
            continue

        write_templated_xhtml(z, name, opts)

def write_toc(z, opts):
    fpath = f'{opts.source_dir}/{opts.toc}'
    if os.path.exists(fpath):
        with open(fpath, 'r') as f:
            file = f.read()
    else:
        items = []
        for i, name in enumerate(opts.spine):
            if len(opts.toc_headings) == 0:
                heading = name
            elif i < len(opts.toc_headings):
                heading = opts.toc_headings[i]
            else:
                continue
            if heading  == '':
                continue

            name_a = quoteattr(name)
            heading_a = xml_escape(heading)
            item = f'<li><a href={name_a}>{heading_a}</a></li>'
            items.append(item)
        file = f'<ol>\n{'\n'.join(items)}\n</ol>'

    nav = opts.template_toc_nav.format(file=file)
    toc = opts.template_toc.format(nav=nav,
        dc_title=xml_escape(opts.dc_title), file=file)

    f = z.open(opts.toc, 'w')
    f.write(toc.encode('utf-8'))
    f.close()

def write_stylesheets(z, opts):
    for name in opts.stylesheets:
        fpath = f'{opts.source_dir}/{name}'
        z.write(fpath, name)

def generate_epub(opts):
    z = zipfile.ZipFile(opts.output, 'w', zipfile.ZIP_STORED)
    z.writestr('mimetype', templates.mimetype)
    z.close()

    z = zipfile.ZipFile(opts.output, 'a', zipfile.ZIP_DEFLATED)
    z.writestr('META-INF/container.xml', templates.container_xml)

    write_package_opf(z, opts)
    write_resource_files(z, opts)
    write_spine_files(z, opts)
    write_toc(z, opts)
    write_stylesheets(z, opts)
    z.close()
