import subprocess

prog = './epubsynth'
out_dir = 'test/out'
src_dir = 'test/src'

def should_pass(output, cmd_args):
    out_path = f'{out_dir}/{output}'
    init_args = [prog, '--output', out_path, '--source-dir', src_dir]
    full_cmd_args = init_args + cmd_args

    print(f'\n\nRunning test case `{output}`')
    print(full_cmd_args)

    result = subprocess.run(full_cmd_args)
    assert result.returncode == 0, f'Test case `{output}` failed'
    result = subprocess.run(['epubcheck', out_path])
    assert result.returncode == 0, f'Test case `{output}` failed'

    print('Passed.')

def should_fail(output, cmd_args):
    out_path = f'{out_dir}/{output}'
    init_args = [prog, '--output', out_path, '--source-dir', src_dir]
    full_cmd_args = init_args + cmd_args

    print(f'\n\nRunning test case `{output}` (error expected)')
    print(full_cmd_args)

    result = subprocess.run(full_cmd_args)
    if result.returncode == 0:
        result = subprocess.run(['epubcheck', out_path])
        assert result.returncode != 0, f'Test case `{output}` failed'

    print('Passed.')

# basic CLI
should_pass('minimal.epub', ['--spine', 'chapter1.xhtml'])
should_fail('spine_missing.epub', [])

# input files
should_pass('basic_complete.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png'])
should_pass('with_css.epub', ['--spine', 'chapter1.xhtml', 'chapter2.xhtml',
            '--resources', 'img.png', '--stylesheets', 'style.css'])
should_fail('missing_image.epub',
            ['--spine', 'chapter1.xhtml', 'chapter2.xhtml'])
should_fail('dup_spine.epub', ['--spine', 'chapter1.xhtml', 'chapter1.xhtml'])
should_fail('dup_resource.epub', ['--spine', 'chapter1.xhtml',
                                  '--resources', 'chapter1.xhtml'])
should_fail('missing_chapter.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', 'chapter3.xhtml', '--resources', 'img.png'])
should_fail('reserved_name1.epub', ['--spine', 'mimetype'])
should_fail('reserved_name2.epub', ['--spine', 'META-INF.xhtml'])
should_fail('reserved_name3.epub', ['--spine', 'package.opf'])
should_fail('wrong_filetype1.epub', ['--spine', 'style.css'])
should_fail('wrong_filetype2.epub', ['--spine', 'img.png'])

# dublin core metadata
should_pass('identifier_given.epub', ['--spine', 'chapter1.xhtml',
                                      '--dc-identifier', '3847582917'])
should_pass('basic_metadata.epub', ['--spine', 'chapter1.xhtml',
    '--dc-title', 'A Set Title', '--dc-creator', 'Maria Nicolae',
    '--dc-language', 'en-AU', '--dc-identifier', '3847582917'])
should_pass('more_metadata.epub', ['--spine', 'chapter1.xhtml',
    '--dc-title', 'A Set Title', '--dc-creator', 'Maria Nicolae',
    '--dc-language', 'en-AU', '--dc-identifier', '3847582917',
    '--dc-contributor', 'Maple Nicolae', 'Ann Author',
    '--dc-rights', 'CC-BY-4.0 OR Apache-2.0'])
should_pass('dc_date_is_now.epub', ['--spine', 'chapter1.xhtml',
    '--dc-identifier', '3847582917', '--dc-date-now'])
should_pass('dc_date_custom.epub', ['--spine', 'chapter1.xhtml',
    '--dc-identifier', '3847582917', '--dc-date', '2012-12-21'])
should_fail('dc_date_invalid.epub', ['--spine', 'chapter1.xhtml',
    '--dc-identifier', '3847582917', '--dc-date', 'December 56, 2012'])
should_fail('dc_date_and_now.epub', ['--spine', 'chapter1.xhtml',
    '--dc-identifier', '3847582917', '--dc-date', '2012-12-21',
    '--dc-date-now'])

# table of contents
should_pass('toc_headings.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc-headings', 'Chapter 1', 'Chapter 2'])
should_pass('toc_headings_only_2.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc-headings', '', 'Chapter 2'])
should_pass('toc_headings_only_1.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc-headings', 'Chapter 1'])
should_pass('toc_custom.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc', 'toc_good.xhtml'])
should_fail('toc_headings_empty.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc-headings'])
should_fail('toc_headings_empty_str.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc-headings', ''])
should_fail('toc_headings_too_many.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc-headings', 'Chapter 1', 'Chapter 2', 'Chapter 3'])
should_fail('toc_empty.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc', 'toc_empty.xhtml'])
should_fail('toc_wrong_type.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc', 'style.css'])

# templating
should_pass('full_xhtml.epub', ['--spine', 'chapter1full.xhtml',
            'chapter2full.xhtml', '--resources', 'img.png',
            '--template-xhtml', '{file}'])
should_pass('full_toc.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc', 'toc_good_full.xhtml', '--template-toc', '{file}'])
should_fail('missing_boilerplate_xhtml.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--template-xhtml', '{file}'])
should_fail('missing_boilerplate_toc.epub', ['--spine', 'chapter1.xhtml',
            'chapter2.xhtml', '--resources', 'img.png',
            '--toc', 'toc_good.xhtml', '--template-toc', '{file}'])

# detect mathml, svg, and javascript
should_pass('special_features.epub',
    ['--spine', 'chapter_mathml.xhtml', 'chapter_script_and_svg.xhtml'])
